// ========================================
// WebSocketサービス - リアルタイム通信管理
// ========================================

import BACKEND_CONFIG from '../config/backend';
import { getCurrentUser } from './auth';

class WebSocketService {
  constructor() {
    this.ws = null;
    this.reconnectAttempts = 0;
    this.reconnectTimer = null;
    this.heartbeatTimer = null;
    this.isConnecting = false;
    this.isDisconnecting = false; // ログアウト時の再接続防止フラグ
    this.listeners = new Map();
    this.connectionStatus = 'disconnected'; // 'disconnected', 'connecting', 'connected'
  }

  /**
   * WebSocket接続を確立
   */
  async connect() {
    // リアルタイム機能が無効の場合
    if (!BACKEND_CONFIG.settings.enableRealtime) {
      console.log('⚠️ リアルタイム機能は無効化されています');
      return false;
    }

    // WebSocket URLが設定されていない場合
    if (BACKEND_CONFIG.websocket.url === 'YOUR_WEBSOCKET_URL') {
      console.warn('⚠️ WebSocket URLが設定されていません');
      return false;
    }

    // すでに接続中の場合
    if (this.ws && this.ws.readyState === WebSocket.OPEN) {
      console.log('✅ WebSocketは既に接続されています');
      return true;
    }

    // 接続処理中の場合
    if (this.isConnecting) {
      console.log('⏳ WebSocket接続処理中...');
      return false;
    }

    try {
      this.isConnecting = true;
      this.connectionStatus = 'connecting';

      // 認証トークンを取得
      let idToken = null;
      try {
        const user = await getCurrentUser();
        idToken = user?.idToken;
      } catch (error) {
        console.warn('⚠️ ユーザー情報の取得に失敗:', error.message);
        this.isConnecting = false;
        return false;
      }

      if (!idToken) {
        console.warn('⚠️ 認証トークンがありません');
        this.isConnecting = false;
        return false;
      }

      // WebSocket URLにトークンを追加
      const wsUrl = `${BACKEND_CONFIG.websocket.url}?token=${idToken}`;

      console.log('🔌 WebSocket接続中...');
      this.ws = new WebSocket(wsUrl);

      // 接続成功時
      this.ws.onopen = () => {
        console.log('✅ WebSocket接続成功');
        this.connectionStatus = 'connected';
        this.isConnecting = false;
        this.reconnectAttempts = 0;

        // ハートビート開始
        this.startHeartbeat();

        // 接続イベントを通知
        this.emit('connected', { timestamp: new Date().toISOString() });
      };

      // メッセージ受信時
      this.ws.onmessage = (event) => {
        try {
          const data = JSON.parse(event.data);
          console.log('📨 WebSocketメッセージ受信:', data);

          // メッセージタイプに応じて処理
          const messageType = data.type || 'message';
          this.emit(messageType, data);

          if (messageType !== 'message') {
            console.log('⚠️ messageイベント以外のタイプでした:', data);
          }
        } catch (error) {
          console.error('❌ メッセージパースエラー:', error);
        }
      };

      // エラー発生時
      this.ws.onerror = (error) => {
        console.error('❌ WebSocketエラー:', error);
        this.emit('error', { error, timestamp: new Date().toISOString() });
      };

      // 接続切断時
      this.ws.onclose = (event) => {
        console.log('🔌 WebSocket切断:', event.code, event.reason);
        this.connectionStatus = 'disconnected';
        this.isConnecting = false;
        this.stopHeartbeat();

        // 切断イベントを通知
        this.emit('disconnected', {
          code: event.code,
          reason: event.reason,
          timestamp: new Date().toISOString()
        });

        // ログアウト中は再接続しない
        if (this.isDisconnecting) {
          console.log('🔌 ログアウトによる切断 - 再接続しません');
          return;
        }

        // 自動再接続（異常切断時のみ）
        if (BACKEND_CONFIG.websocket.reconnect.enabled && !event.wasClean) {
          this.scheduleReconnect();
        }
      };

      return true;
    } catch (error) {
      console.error('❌ WebSocket接続エラー:', error);
      this.isConnecting = false;
      this.connectionStatus = 'disconnected';
      return false;
    }
  }

  /**
   * WebSocket切断
   */
  disconnect() {
    console.log('🔌 WebSocket切断中...');

    // ログアウト中フラグを立てる（再接続を防止）
    this.isDisconnecting = true;

    // 再接続タイマーをクリア
    if (this.reconnectTimer) {
      clearTimeout(this.reconnectTimer);
      this.reconnectTimer = null;
    }

    // ハートビートタイマーをクリア
    this.stopHeartbeat();

    // WebSocket切断
    if (this.ws) {
      this.ws.close(1000, 'Client disconnect');
      this.ws = null;
    }

    // イベントリスナーをクリア
    this.listeners.clear();

    this.connectionStatus = 'disconnected';
    this.reconnectAttempts = 0;

    // フラグをリセット（1秒後に再接続可能にする）
    setTimeout(() => {
      this.isDisconnecting = false;
    }, 1000);

    console.log('✅ WebSocket切断完了');
  }

  /**
   * メッセージ送信
   */
  send(action, data) {
    if (!this.isConnected()) {
      console.warn('⚠️ WebSocketが接続されていません');
      return false;
    }

    try {
      const message = JSON.stringify({
        action,
        data,
        timestamp: new Date().toISOString()
      });

      this.ws.send(message);
      console.log('📤 メッセージ送信:', action, data);
      return true;
    } catch (error) {
      console.error('❌ メッセージ送信エラー:', error);
      return false;
    }
  }

  /**
   * チャットメッセージ送信
   */
  sendMessage(message) {
    return this.send('send_message', { message });
  }

  /**
   * 位置情報更新を送信
   */
  sendLocationUpdate(latitude, longitude, memo) {
    return this.send('location_update', { latitude, longitude, memo });
  }

  /**
   * イベントリスナーを登録
   */
  on(event, callback) {
    if (!this.listeners.has(event)) {
      this.listeners.set(event, []);
    }
    this.listeners.get(event).push(callback);

    // リスナー解除関数を返す
    return () => {
      const callbacks = this.listeners.get(event);
      if (callbacks) {
        const index = callbacks.indexOf(callback);
        if (index !== -1) {
          callbacks.splice(index, 1);
        }
      }
    };
  }

  /**
   * イベントリスナーを解除
   */
  off(event, callback) {
    const callbacks = this.listeners.get(event);
    if (callbacks) {
      const index = callbacks.indexOf(callback);
      if (index !== -1) {
        callbacks.splice(index, 1);
      }
    }
  }

  /**
   * イベントを発火
   */
  emit(event, data) {
    const callbacks = this.listeners.get(event);
    if (callbacks) {
      callbacks.forEach(callback => {
        try {
          callback(data);
        } catch (error) {
          console.error(`❌ イベントコールバックエラー (${event}):`, error);
        }
      });
    }
  }

  /**
   * 再接続をスケジュール
   */
  scheduleReconnect() {
    const { maxAttempts, interval } = BACKEND_CONFIG.websocket.reconnect;

    if (this.reconnectAttempts >= maxAttempts) {
      console.error('❌ WebSocket再接続の最大試行回数に達しました');
      this.emit('reconnect_failed', { attempts: this.reconnectAttempts });
      return;
    }

    this.reconnectAttempts++;
    const delay = interval * this.reconnectAttempts;

    console.log(`🔄 WebSocket再接続を${delay}ms後に試行します (${this.reconnectAttempts}/${maxAttempts})`);

    this.reconnectTimer = setTimeout(() => {
      console.log(`🔄 WebSocket再接続試行中... (${this.reconnectAttempts}/${maxAttempts})`);
      this.connect();
    }, delay);
  }

  /**
   * ハートビート開始（接続維持）
   */
  startHeartbeat() {
    if (!BACKEND_CONFIG.websocket.heartbeat.enabled) {
      return;
    }

    this.stopHeartbeat();

    this.heartbeatTimer = setInterval(() => {
      if (this.isConnected()) {
        this.send('ping', { timestamp: new Date().toISOString() });
      }
    }, BACKEND_CONFIG.websocket.heartbeat.interval);

    console.log('💓 ハートビート開始');
  }

  /**
   * ハートビート停止
   */
  stopHeartbeat() {
    if (this.heartbeatTimer) {
      clearInterval(this.heartbeatTimer);
      this.heartbeatTimer = null;
      console.log('💓 ハートビート停止');
    }
  }

  /**
   * 接続状態を確認
   */
  isConnected() {
    return this.ws && this.ws.readyState === WebSocket.OPEN;
  }

  /**
   * 接続状態を取得
   */
  getConnectionStatus() {
    return this.connectionStatus;
  }

  /**
   * 再接続試行回数を取得
   */
  getReconnectAttempts() {
    return this.reconnectAttempts;
  }
}

// シングルトンインスタンスをエクスポート
const websocketService = new WebSocketService();

export default websocketService;

// 名前付きエクスポート（テスト用）
export { WebSocketService };

