import {
  CognitoUserPool,
  CognitoUser,
  AuthenticationDetails,
  CognitoUserAttribute,
} from 'amazon-cognito-identity-js';
import AsyncStorage from '@react-native-async-storage/async-storage';
import BACKEND_CONFIG from '../config/backend';

// Cognito User Pool設定
const poolData = {
  UserPoolId: BACKEND_CONFIG.cognito.userPoolId,
  ClientId: BACKEND_CONFIG.cognito.clientId,
};

const userPool = new CognitoUserPool(poolData);

// ========================================
// ユーザー登録
// ========================================
export const signUp = (email, password) => {
  return new Promise((resolve, reject) => {
    const attributeList = [
      new CognitoUserAttribute({
        Name: 'email',
        Value: email,
      }),
    ];

    userPool.signUp(email, password, attributeList, null, (err, result) => {
      if (err) {
        reject(err);
        return;
      }
      resolve(result.user);
    });
  });
};

// ========================================
// メール認証コード確認
// ========================================
export const confirmSignUp = (email, code) => {
  return new Promise((resolve, reject) => {
    const cognitoUser = new CognitoUser({
      Username: email,
      Pool: userPool,
    });

    cognitoUser.confirmRegistration(code, true, (err, result) => {
      if (err) {
        reject(err);
        return;
      }
      resolve(result);
    });
  });
};

// ========================================
// ログイン
// ========================================
export const signIn = (email, password) => {
  console.log('🔵 [AUTH] signIn() 開始:', email);

  return new Promise((resolve, reject) => {
    const authenticationDetails = new AuthenticationDetails({
      Username: email,
      Password: password,
    });

    const cognitoUser = new CognitoUser({
      Username: email,
      Pool: userPool,
    });

    console.log('🔵 [AUTH] Cognito認証を実行中...');

    cognitoUser.authenticateUser(authenticationDetails, {
      onSuccess: async (result) => {
        console.log('🔵 [AUTH] Cognito認証成功');

        const idToken = result.getIdToken().getJwtToken();
        const accessToken = result.getAccessToken().getJwtToken();

        console.log('🔵 [AUTH] トークンをAsyncStorageに保存中...');

        // トークンを保存
        await AsyncStorage.setItem('idToken', idToken);
        await AsyncStorage.setItem('accessToken', accessToken);
        await AsyncStorage.setItem('userEmail', email);

        console.log('✅ [AUTH] トークン保存完了');

        resolve({
          idToken,
          accessToken,
          email,
        });
      },
      onFailure: (err) => {
        console.error('❌ [AUTH] Cognito認証失敗:', err);
        reject(err);
      },
    });
  });
};

// ========================================
// ログアウト
// ========================================
export const signOut = async () => {
  const cognitoUser = userPool.getCurrentUser();
  if (cognitoUser) {
    cognitoUser.signOut();
  }

  // トークンを削除
  await AsyncStorage.removeItem('idToken');
  await AsyncStorage.removeItem('accessToken');
  await AsyncStorage.removeItem('userEmail');
};

// ========================================
// 現在のユーザーを取得
// ========================================
export const getCurrentUser = async () => {
  try {
    // AsyncStorage からトークンとユーザー情報を取得
    const idToken = await AsyncStorage.getItem('idToken');
    const accessToken = await AsyncStorage.getItem('accessToken');
    const userEmail = await AsyncStorage.getItem('userEmail');

    if (!idToken) {
      throw new Error('No user found');
    }

    return {
      idToken,
      accessToken,
      email: userEmail,
    };
  } catch (error) {
    throw new Error('No user found');
  }
};

// ========================================
// トークンを取得
// ========================================
export const getIdToken = async () => {
  const token = await AsyncStorage.getItem('idToken');
  return token;
};

// ========================================
// ログイン状態をチェック
// ========================================
export const isAuthenticated = async () => {
  const token = await AsyncStorage.getItem('idToken');
  return token !== null;
};
