import axios from 'axios';
import BACKEND_CONFIG from '../config/backend';
import { getIdToken } from './auth';

// APIクライアントのベース設定
const apiClient = axios.create({
  baseURL: BACKEND_CONFIG.api.baseUrl,
  timeout: 10000,
  headers: {
    'Content-Type': 'application/json',
  },
});

// リクエストインターセプター（認証トークンを自動で追加）
apiClient.interceptors.request.use(
  async (config) => {
    const token = await getIdToken();
    if (token) {
      config.headers.Authorization = `Bearer ${token}`;
    }
    return config;
  },
  (error) => {
    return Promise.reject(error);
  }
);

// レスポンスインターセプター（エラーハンドリング）
apiClient.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response) {
      // サーバーがエラーレスポンスを返した
      console.error('API Error:', error.response.status, error.response.data);
    } else if (error.request) {
      // リクエストは送信されたがレスポンスがない
      console.error('Network Error:', error.request);
    } else {
      // リクエスト設定中にエラーが発生
      console.error('Error:', error.message);
    }
    return Promise.reject(error);
  }
);

// ========================================
// チェックイン作成
// ========================================
export const createCheckIn = async (latitude, longitude, memo = '') => {
  try {
    const response = await apiClient.post(
      BACKEND_CONFIG.api.endpoints.createCheckIn,
      {
        latitude,
        longitude,
        memo,
        timestamp: new Date().toISOString(),
      }
    );
    return response.data;
  } catch (error) {
    throw new Error(`チェックインに失敗しました: ${error.message}`);
  }
};

// ========================================
// チェックイン履歴取得
// ========================================
export const getCheckIns = async (limit = BACKEND_CONFIG.settings.historyLimit) => {
  try {
    const response = await apiClient.get(
      BACKEND_CONFIG.api.endpoints.getCheckIns,
      {
        params: { limit },
      }
    );
    return response.data;
  } catch (error) {
    throw new Error(`履歴の取得に失敗しました: ${error.message}`);
  }
};

// ========================================
// 近くのチェックイン取得
// ========================================
export const getNearbyCheckIns = async (latitude, longitude, radiusKm = BACKEND_CONFIG.settings.searchRadiusKm) => {
  try {
    const response = await apiClient.get(
      BACKEND_CONFIG.api.endpoints.getNearbyCheckIns,
      {
        params: {
          latitude,
          longitude,
          radius: radiusKm,
        },
      }
    );
    return response.data;
  } catch (error) {
    throw new Error(`近くのチェックインの取得に失敗しました: ${error.message}`);
  }
};

// ========================================
// チェックイン削除（オプション）
// ========================================
export const deleteCheckIn = async (checkInId) => {
  try {
    const response = await apiClient.delete(
      `${BACKEND_CONFIG.api.endpoints.createCheckIn}/${checkInId}`
    );
    return response.data;
  } catch (error) {
    throw new Error(`チェックインの削除に失敗しました: ${error.message}`);
  }
};

// ========================================
// ヘルスチェック（APIの疎通確認）
// ========================================
export const healthCheck = async () => {
  try {
    const response = await apiClient.get('/health');
    return response.data;
  } catch (error) {
    throw new Error(`APIに接続できません: ${error.message}`);
  }
};

export default apiClient;
