// ========================================
// バックエンド設定ファイル
// ここにAWSのエンドポイント情報を設定してください
// ========================================

const BACKEND_CONFIG = {
  // ========================================
  // Amazon Cognito 設定
  // ========================================
  cognito: {
    // Cognito User Pool ID
    // 例: 'us-east-1_xxxxxxxxx'
    userPoolId: 'us-east-1_ABCD',

    // Cognito App Client ID
    // 例: '1234567890abcdefghijklmnop'
    clientId: 'ABCDEFGHIJ',

    // AWSリージョン
    // 例: 'us-east-1' (バージニア北部リージョン)
    region: 'us-east-1',
  },

  // ========================================
  // API Gateway 設定
  // ========================================
  api: {
    // API GatewayのベースURL
    // 例: 'https://xxxxxxxxxx.execute-api.us-east-1.amazonaws.com/prod'
    baseUrl: 'https://abcdefghij.execute-api.us-east-1.amazonaws.com/prod',

    // APIエンドポイント
    endpoints: {
      // チェックイン作成
      createCheckIn: '/checkin',

      // チェックイン履歴取得
      getCheckIns: '/checkins',

      // 近くのチェックイン取得
      getNearbyCheckIns: '/checkins/nearby',
    },
  },

  // ========================================
  // WebSocket API 設定 (Phase 2)
  // ========================================
  websocket: {
    // WebSocket API URL
    // 例: 'wss://xxxxxxxxxx.execute-api.us-east-1.amazonaws.com/prod'
    url: 'wss://abcdefghij.execute-api.us-east-1.amazonaws.com/prod/',

    // 再接続設定
    reconnect: {
      enabled: true,
      maxAttempts: 5,
      interval: 3000, // ミリ秒
    },

    // ハートビート設定（接続維持）
    heartbeat: {
      enabled: true,
      interval: 30000, // 30秒ごとにpingを送信
    },
  },

  // ========================================
  // その他の設定
  // ========================================
  settings: {
    // 近くのチェックインを検索する範囲（km）
    searchRadiusKm: 5,

    // チェックイン履歴の取得件数
    historyLimit: 50,

    // リアルタイム更新の有効化 (Phase 2)
    enableRealtime: true,
  },
};

export default BACKEND_CONFIG;

// ========================================
// 設定の検証
// ========================================
export const validateConfig = () => {
  const errors = [];
  const warnings = [];

  if (BACKEND_CONFIG.cognito.userPoolId === 'YOUR_USER_POOL_ID') {
    errors.push('⚠️ Cognito User Pool IDが設定されていません');
  }

  if (BACKEND_CONFIG.cognito.clientId === 'YOUR_CLIENT_ID') {
    errors.push('⚠️ Cognito Client IDが設定されていません');
  }

  if (BACKEND_CONFIG.api.baseUrl === 'YOUR_API_GATEWAY_URL') {
    errors.push('⚠️ API Gateway URLが設定されていません');
  }

  // Phase 2: WebSocket設定の検証
  if (BACKEND_CONFIG.settings.enableRealtime) {
    if (BACKEND_CONFIG.websocket.url === 'YOUR_WEBSOCKET_URL') {
      warnings.push('⚠️ WebSocket URLが設定されていません（リアルタイム機能は無効化されます）');
    }
  }

  if (errors.length > 0) {
    console.warn('=== バックエンド設定エラー ===');
    errors.forEach(error => console.warn(error));
    console.warn('src/config/backend.js を編集してください');
    return false;
  }

  if (warnings.length > 0) {
    console.warn('=== バックエンド設定警告 ===');
    warnings.forEach(warning => console.warn(warning));
  }

  console.log('✅ バックエンド設定OK');
  return true;
};
