import React, { useState } from 'react';
import {
  View,
  Text,
  TextInput,
  TouchableOpacity,
  StyleSheet,
  Alert,
  ActivityIndicator,
  KeyboardAvoidingView,
  Platform,
  Keyboard,
} from 'react-native';
import { useRouter } from 'expo-router';
import { signIn, signUp, confirmSignUp } from '../services/auth';

export default function LoginScreen() {
  const router = useRouter();
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [confirmationCode, setConfirmationCode] = useState('');
  const [isSignUp, setIsSignUp] = useState(false);
  const [needsConfirmation, setNeedsConfirmation] = useState(false);
  const [loading, setLoading] = useState(false);

  // ========================================
  // ログイン処理
  // ========================================
  const handleSignIn = async () => {
    console.log('🔵 [LOGIN] ログインボタンがタップされました');

    // キーボードを閉じる
    Keyboard.dismiss();

    if (!email || !password) {
      console.log('⚠️ [LOGIN] 入力チェック失敗: email or password が空');
      Alert.alert('エラー', 'メールアドレスとパスワードを入力してください');
      return;
    }

    console.log('🔵 [LOGIN] ログイン処理を開始:', email);
    setLoading(true);

    try {
      console.log('🔵 [LOGIN] signIn() を呼び出し中...');
      const result = await signIn(email, password);
      console.log('✅ [LOGIN] ログイン成功:', result);

      // ログイン成功後、メイン画面にリダイレクト
      console.log('🔵 [LOGIN] メイン画面にリダイレクト');
      router.replace('/');
    } catch (error) {
      console.error('❌ [LOGIN] ログイン失敗:', error);
      Alert.alert('ログイン失敗', error.message);
    } finally {
      console.log('🔵 [LOGIN] ログイン処理完了（loading = false）');
      setLoading(false);
    }
  };

  // ========================================
  // ユーザー登録処理
  // ========================================
  const handleSignUp = async () => {
    // キーボードを閉じる
    Keyboard.dismiss();

    if (!email || !password) {
      Alert.alert('エラー', 'メールアドレスとパスワードを入力してください');
      return;
    }

    if (password.length < 8) {
      Alert.alert('エラー', 'パスワードは8文字以上にしてください');
      return;
    }

    setLoading(true);
    try {
      await signUp(email, password);
      setNeedsConfirmation(true);
      Alert.alert(
        '確認コード送信',
        'メールアドレスに確認コードを送信しました。コードを入力してください。'
      );
    } catch (error) {
      Alert.alert('登録失敗', error.message);
    } finally {
      setLoading(false);
    }
  };

  // ========================================
  // メール確認処理
  // ========================================
  const handleConfirmSignUp = async () => {
    // キーボードを閉じる
    Keyboard.dismiss();

    if (!confirmationCode) {
      Alert.alert('エラー', '確認コードを入力してください');
      return;
    }

    setLoading(true);
    try {
      await confirmSignUp(email, confirmationCode);
      Alert.alert('登録完了', 'ユーザー登録が完了しました。ログインしてください。', [
        {
          text: 'OK',
          onPress: () => {
            setNeedsConfirmation(false);
            setIsSignUp(false);
            setConfirmationCode('');
          },
        },
      ]);
    } catch (error) {
      Alert.alert('確認失敗', error.message);
    } finally {
      setLoading(false);
    }
  };

  // ========================================
  // メール確認画面
  // ========================================
  if (needsConfirmation) {
    return (
      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        style={styles.container}>
        <View style={styles.formContainer}>
          <Text style={styles.title}>メール確認</Text>
          <Text style={styles.subtitle}>
            {email} に送信された確認コードを入力してください
          </Text>

          <TextInput
            style={styles.input}
            placeholder="確認コード"
            value={confirmationCode}
            onChangeText={setConfirmationCode}
            keyboardType="number-pad"
            autoCapitalize="none"
            autoComplete="one-time-code"
            textContentType="oneTimeCode"
            returnKeyType="done"
            onSubmitEditing={handleConfirmSignUp}
          />

          <TouchableOpacity
            style={styles.button}
            onPress={handleConfirmSignUp}
            disabled={loading}>
            {loading ? (
              <ActivityIndicator color="#fff" />
            ) : (
              <Text style={styles.buttonText}>確認</Text>
            )}
          </TouchableOpacity>

          <TouchableOpacity
            style={styles.linkButton}
            onPress={() => setNeedsConfirmation(false)}>
            <Text style={styles.linkText}>戻る</Text>
          </TouchableOpacity>
        </View>
      </KeyboardAvoidingView>
    );
  }

  // ========================================
  // ログイン/登録画面
  // ========================================
  return (
    <KeyboardAvoidingView
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      style={styles.container}>
      <View style={styles.formContainer}>
        <Text style={styles.title}>📍 TENARAI Blueprint - Sample Mobile App</Text>
        <Text style={styles.subtitle}>
          {isSignUp ? 'アカウント作成' : 'ログイン'}
        </Text>

        <TextInput
          style={styles.input}
          placeholder="メールアドレス"
          value={email}
          onChangeText={setEmail}
          keyboardType="email-address"
          autoCapitalize="none"
          autoCorrect={false}
          autoComplete="email"
          textContentType="emailAddress"
          returnKeyType="next"
        />

        <TextInput
          style={styles.input}
          placeholder="パスワード"
          value={password}
          onChangeText={setPassword}
          secureTextEntry
          autoCapitalize="none"
          autoComplete="password"
          textContentType="password"
          returnKeyType="done"
          onSubmitEditing={isSignUp ? handleSignUp : handleSignIn}
        />

        <TouchableOpacity
          style={styles.button}
          onPress={isSignUp ? handleSignUp : handleSignIn}
          disabled={loading}>
          {loading ? (
            <ActivityIndicator color="#fff" />
          ) : (
            <Text style={styles.buttonText}>
              {isSignUp ? '登録' : 'ログイン'}
            </Text>
          )}
        </TouchableOpacity>

        <TouchableOpacity
          style={styles.linkButton}
          onPress={() => setIsSignUp(!isSignUp)}>
          <Text style={styles.linkText}>
            {isSignUp
              ? 'すでにアカウントをお持ちですか？ ログイン'
              : 'アカウントをお持ちでない方は こちら'}
          </Text>
        </TouchableOpacity>
      </View>
    </KeyboardAvoidingView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
    justifyContent: 'center',
  },
  formContainer: {
    padding: 20,
    backgroundColor: '#fff',
    margin: 20,
    borderRadius: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    textAlign: 'center',
    marginBottom: 10,
    color: '#333',
  },
  subtitle: {
    fontSize: 16,
    textAlign: 'center',
    marginBottom: 20,
    color: '#666',
  },
  input: {
    backgroundColor: '#f9f9f9',
    padding: 15,
    borderRadius: 8,
    marginBottom: 15,
    fontSize: 16,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  button: {
    backgroundColor: '#007AFF',
    padding: 15,
    borderRadius: 8,
    alignItems: 'center',
    marginTop: 10,
  },
  buttonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  linkButton: {
    marginTop: 15,
    alignItems: 'center',
  },
  linkText: {
    color: '#007AFF',
    fontSize: 14,
  },
});

