import React, { useState, useEffect, useCallback, useRef } from 'react';
import {
  View,
  Text,
  TouchableOpacity,
  StyleSheet,
  Alert,
  ActivityIndicator,
  ScrollView,
} from 'react-native';
import { useRouter } from 'expo-router';
import * as Location from 'expo-location';
import { createCheckIn, getCheckIns } from '../services/api';
import { signOut, isAuthenticated } from '../services/auth';
import websocketService from '../services/websocket';
import BACKEND_CONFIG from '../config/backend';

export default function MapScreen() {
  const router = useRouter();
  const [currentLocation, setCurrentLocation] = useState(null);
  const [checkIns, setCheckIns] = useState([]);
  const [loading, setLoading] = useState(false);
  const [isReady, setIsReady] = useState(false);

  // Phase 2: WebSocket関連のstate
  const [wsConnected, setWsConnected] = useState(false);
  const [realtimeCheckIns, setRealtimeCheckIns] = useState([]);
  const [realtimeEnabled] = useState(BACKEND_CONFIG.settings.enableRealtime);
  const [wsMessages, setWsMessages] = useState([]);

  // ========================================
  // 初期化：画面マウント時に実行
  // ========================================
  useEffect(() => {
    console.log('🔵 [MAIN] メイン画面がマウントされました');

    const initialize = async () => {
      // 認証状態を確認
      console.log('🔵 [MAIN] 認証状態を確認中...');
      const authenticated = await isAuthenticated();
      console.log('🔵 [MAIN] 認証状態:', authenticated ? '認証済み' : '未認証');

      if (!authenticated) {
        // 未認証の場合はログイン画面にリダイレクト
        console.log('⚠️ [MAIN] 未認証のためログイン画面にリダイレクト');
        router.replace('/login');
        return;
      }

      console.log('✅ [MAIN] メイン画面初期化を開始');

      // 位置情報とチェックイン履歴を取得
      console.log('🔵 [MAIN] 位置情報を取得中...');
      getCurrentLocation();

      console.log('🔵 [MAIN] チェックイン履歴を取得中...');
      loadCheckIns();

      // Phase 2: WebSocket接続
      if (realtimeEnabled) {
        console.log('🔵 [MAIN] WebSocket接続を開始...');
        initializeWebSocket();
      }

      setIsReady(true);
      console.log('✅ [MAIN] 初期化完了');
    };

    initialize();

    // クリーンアップ
    return () => {
      console.log('🔵 [MAIN] メイン画面がアンマウントされました');
      if (realtimeEnabled) {
        websocketService.disconnect();
      }
    };
  }, []);

  // ========================================
  // Phase 2: WebSocket初期化とイベントリスナー設定
  // ========================================
  const initializeWebSocket = async () => {
    try {
      // WebSocket接続
      const connected = await websocketService.connect();

      if (!connected) {
        console.log('⚠️ WebSocket接続に失敗しました');
        return;
      }

      // 接続成功イベント
      websocketService.on('connected', () => {
        console.log('✅ WebSocket接続成功');
        setWsConnected(true);
      });

      // 切断イベント
      websocketService.on('disconnected', () => {
        console.log('🔌 WebSocket切断');
        setWsConnected(false);
      });

      // 新しいチェックイン通知を受信
      websocketService.on('notification', (data) => {
        handleRealtimeNotification(data);
      });

      // リアルタイムメッセージを受信
      websocketService.on('message', (data) => {
        console.log('📨 リアルタイムメッセージ:', data);

        // メッセージを保存（最新3件のみ保持）
        setWsMessages((prev) => {
          const newMessage = {
            ...data,
            timestamp: data.timestamp || new Date().toISOString(),
            id: `msg-${Date.now()}-${Math.random()}`
          };
          return [newMessage, ...prev].slice(0, 3);
        });
      });

      // エラーイベント
      websocketService.on('error', (error) => {
        console.error('❌ WebSocketエラー:', error);
      });

    } catch (error) {
      console.error('❌ WebSocket初期化エラー:', error);
    }
  };

  // ========================================
  // Phase 2: リアルタイム通知を処理
  // ========================================
  const handleRealtimeNotification = useCallback((data) => {
    console.log('📨 リアルタイム通知受信:', data);

    if (data.event === 'newCheckIn' && data.data) {
      const newCheckIn = data.data;

      // リアルタイムチェックインリストに追加（重複チェック）
      setRealtimeCheckIns((prev) => {
        const exists = prev.some(
          (item) => item.checkInId === newCheckIn.checkInId
        );
        if (exists) return prev;

        // 最新のものを先頭に追加（最大10件まで保持）
        return [newCheckIn, ...prev].slice(0, 10);
      });

      // 通知を表示（自分のチェックインは除外）
      if (newCheckIn.userId !== 'current-user-id') { // TODO: 実際のユーザーIDで判定
        Alert.alert(
          '📍 新しいチェックイン',
          `近くで誰かがチェックインしました！\n${newCheckIn.memo || ''}`,
          [{ text: 'OK' }]
        );
      }
    }
  }, []);

  // ========================================
  // 現在地を取得
  // ========================================
  const getCurrentLocation = async () => {
    try {
      // 位置情報の権限をリクエスト
      const { status } = await Location.requestForegroundPermissionsAsync();
      if (status !== 'granted') {
        Alert.alert('エラー', '位置情報の権限が必要です');
        // デフォルト位置（東京）
        setCurrentLocation({
          latitude: 35.6812,
          longitude: 139.7671,
        });
        return;
      }

      // 現在地を取得
      const location = await Location.getCurrentPositionAsync({
        accuracy: Location.Accuracy.High,
      });

      setCurrentLocation({
        latitude: location.coords.latitude,
        longitude: location.coords.longitude,
      });
    } catch (error) {
        Alert.alert('エラー', '位置情報の取得に失敗しました');
        console.error(error);
        // デフォルト位置（東京）
        setCurrentLocation({
          latitude: 35.6812,
          longitude: 139.7671,
        });
    }
  };

  // ========================================
  // チェックイン履歴を読み込み
  // ========================================
  const loadCheckIns = async () => {
    try {
      const data = await getCheckIns();
      // データ形式に応じて調整
      const checkInList = Array.isArray(data) ? data : data.items || [];
      setCheckIns(checkInList);
    } catch (error) {
      console.error('履歴の読み込みエラー:', error);
      // エラーでもアプリは動作続行
    }
  };

  // ========================================
  // チェックイン実行
  // ========================================
  const handleCheckIn = async () => {
    if (!currentLocation) {
      Alert.alert('エラー', '位置情報を取得できていません');
      return;
    }

    setLoading(true);
    try {
      const result = await createCheckIn(
        currentLocation.latitude,
        currentLocation.longitude,
        '📍 チェックイン'
      );
      Alert.alert('成功', 'チェックインしました！');
      loadCheckIns(); // 履歴を再読み込み

      // Phase 2: WebSocketで位置情報更新を送信
      if (realtimeEnabled && wsConnected) {
        websocketService.sendLocationUpdate(
          currentLocation.latitude,
          currentLocation.longitude,
          '📍 チェックイン'
        );
      }
    } catch (error) {
      Alert.alert('エラー', error.message);
    } finally {
      setLoading(false);
    }
  };

  // ========================================
  // Say Hi!メッセージを送信
  // ========================================
  const handleSayHi = () => {
    if (!wsConnected) {
      Alert.alert('エラー', 'WebSocketが接続されていません');
      return;
    }

    try {
      websocketService.sendMessage('👋 Hi! みんな元気？');
      Alert.alert('送信完了', 'メッセージを送信しました！');
    } catch (error) {
      Alert.alert('エラー', 'メッセージの送信に失敗しました');
      console.error('メッセージ送信エラー:', error);
    }
  };

  // ========================================
  // ログアウト
  // ========================================
  const handleLogout = () => {
    console.log('🔵 [MAIN] ログアウトボタンがタップされました');

    Alert.alert('ログアウト', '本当にログアウトしますか？', [
      {
        text: 'キャンセル',
        style: 'cancel',
        onPress: () => console.log('🔵 [MAIN] ログアウトをキャンセル')
      },
      {
        text: 'ログアウト',
        onPress: async () => {
          console.log('🔵 [MAIN] ログアウト処理を開始');

          // トークンを削除
          console.log('🔵 [MAIN] トークンを削除中...');
          await signOut();
          console.log('✅ [MAIN] トークン削除完了');

          // Phase 2: WebSocket切断
          if (realtimeEnabled) {
            console.log('🔵 [MAIN] WebSocket切断中...');
            websocketService.disconnect();
          }

          // ログイン画面にリダイレクト
          console.log('🔵 [MAIN] ログイン画面にリダイレクト');
          router.replace('/login');
        },
      },
    ]);
  };

  // ========================================
  // 日時フォーマット
  // ========================================
  const formatDate = (timestamp) => {
    const date = new Date(timestamp);
    return `${date.getFullYear()}/${date.getMonth() + 1}/${date.getDate()} ${date.getHours()}:${String(
      date.getMinutes()
    ).padStart(2, '0')}`;
  };

  // ========================================
  // チェックインアイテムをレンダリング
  // ========================================
  const renderCheckInItem = ({ item, index, isRealtime }) => (
    <View style={[styles.checkInItem, isRealtime && styles.realtimeItem]}>
      <View style={styles.checkInHeader}>
        <Text style={styles.checkInLabel}>
          {isRealtime ? '🟢 リアルタイム' : '📍 チェックイン'}
        </Text>
        <Text style={styles.checkInDate}>{formatDate(item.timestamp)}</Text>
      </View>

      <View style={styles.infoRow}>
        <Text style={styles.infoLabel}>ユーザーID:</Text>
        <Text style={styles.infoValue}>{item.userId || 'unknown'}</Text>
      </View>

      <View style={styles.infoRow}>
        <Text style={styles.infoLabel}>緯度:</Text>
        <Text style={styles.infoValue}>{item.latitude?.toFixed(6) || 'N/A'}</Text>
      </View>

      <View style={styles.infoRow}>
        <Text style={styles.infoLabel}>経度:</Text>
        <Text style={styles.infoValue}>{item.longitude?.toFixed(6) || 'N/A'}</Text>
      </View>

      {item.memo && (
        <View style={styles.infoRow}>
          <Text style={styles.infoLabel}>メモ:</Text>
          <Text style={styles.infoValue}>{item.memo}</Text>
        </View>
      )}

      {item.checkInId && (
        <View style={styles.infoRow}>
          <Text style={styles.infoLabel}>ID:</Text>
          <Text style={[styles.infoValue, styles.checkInId]}>{item.checkInId}</Text>
        </View>
      )}
    </View>
  );

  // ========================================
  // レンダリング
  // ========================================
  if (!isReady || !currentLocation) {
    return (
      <View style={styles.loadingContainer}>
        <ActivityIndicator size="large" color="#007AFF" />
        <Text style={styles.loadingText}>
          {!isReady ? '読み込み中...' : '位置情報を取得中...'}
        </Text>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      {/* ヘッダー */}
      <View style={styles.header}>
        <View style={styles.titleContainer}>
          <Text style={styles.title}>📍 チェックイン</Text>
          {/* Phase 2: WebSocket接続状態インジケーター */}
          {realtimeEnabled && (
            <View style={styles.wsStatusContainer}>
              <View
                style={[
                  styles.wsStatusDot,
                  { backgroundColor: wsConnected ? '#4CAF50' : '#FF9800' }
                ]}
              />
              <Text style={styles.wsStatusText}>
                {wsConnected ? 'リアルタイム' : '接続中...'}
              </Text>
            </View>
          )}
        </View>
        <TouchableOpacity onPress={handleLogout} style={styles.logoutButton}>
          <Text style={styles.logoutText}>ログアウト</Text>
        </TouchableOpacity>
      </View>

      {/* WebSocketメッセージ表示 */}
      {realtimeEnabled && wsMessages.length > 0 && (
        <View style={styles.messagesCard}>
          <Text style={styles.cardTitle}>💬 受信メッセージ (最新3件)</Text>
          {wsMessages.map((msg, index) => (
            <View key={msg.id} style={styles.messageItem}>
              <View style={styles.messageHeader}>
                <Text style={styles.messageIndex}>#{index + 1}</Text>
                <Text style={styles.messageTime}>
                  {new Date(msg.timestamp).toLocaleTimeString('ja-JP')}
                </Text>
              </View>
              {/* User ID表示 */}
              {(msg.sender_id || msg.data?.sender_id) && (
                <View style={styles.messageUserRow}>
                  <Text style={styles.messageUserLabel}>User ID:</Text>
                  <Text style={styles.messageUserValue}>
                    {msg.sender_id || msg.data?.sender_id}
                  </Text>
                </View>
              )}
              <Text style={styles.messageContent}>
                {msg.data?.message || msg.message || JSON.stringify(msg, null, 2)}
              </Text>
            </View>
          ))}
        </View>
      )}

      {/* 現在地表示 */}
      <View style={styles.currentLocationCard}>
        <Text style={styles.cardTitle}>📍 現在地</Text>
        <View style={styles.infoRow}>
          <Text style={styles.infoLabel}>緯度:</Text>
          <Text style={styles.infoValue}>{currentLocation.latitude.toFixed(6)}</Text>
        </View>
        <View style={styles.infoRow}>
          <Text style={styles.infoLabel}>経度:</Text>
          <Text style={styles.infoValue}>{currentLocation.longitude.toFixed(6)}</Text>
        </View>
      </View>

      {/* ボタンエリア */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity
          style={[styles.checkInButton, loading && styles.buttonDisabled]}
          onPress={handleCheckIn}
          disabled={loading}>
          {loading ? (
            <ActivityIndicator color="#fff" />
          ) : (
            <Text style={styles.checkInButtonText}>チェックイン</Text>
          )}
        </TouchableOpacity>

        <TouchableOpacity
          style={styles.refreshButton}
          onPress={() => {
            getCurrentLocation();
            loadCheckIns();
          }}>
          <Text style={styles.refreshButtonText}>🔄 更新</Text>
        </TouchableOpacity>
      </View>

      {/* Say Hi!ボタン（WebSocket有効時のみ表示） */}
      {realtimeEnabled && (
        <View style={styles.buttonContainer}>
          <TouchableOpacity
            style={[styles.sayHiButton, !wsConnected && styles.buttonDisabled]}
            onPress={handleSayHi}
            disabled={!wsConnected}>
            <Text style={styles.sayHiButtonText}>👋 Say Hi!</Text>
          </TouchableOpacity>
        </View>
      )}

      {/* チェックインリスト */}
      <ScrollView style={styles.listContainer}>
        {/* リアルタイムチェックイン */}
        {realtimeEnabled && realtimeCheckIns.length > 0 && (
          <View style={styles.section}>
            <Text style={styles.sectionTitle}>🟢 リアルタイムチェックイン</Text>
            {realtimeCheckIns.map((item, index) => (
              <View key={`realtime-${item.checkInId || index}`}>
                {renderCheckInItem({ item, index, isRealtime: true })}
              </View>
            ))}
          </View>
        )}

        {/* チェックイン履歴 */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>
            📋 チェックイン履歴 ({checkIns.length}件)
          </Text>
          {checkIns.length > 0 ? (
            checkIns.map((item, index) => (
              <View key={`checkin-${item.checkInId || item.timestamp}-${index}`}>
                {renderCheckInItem({ item, index, isRealtime: false })}
              </View>
            ))
          ) : (
            <Text style={styles.emptyText}>チェックイン履歴がありません</Text>
          )}
        </View>
      </ScrollView>
    </View>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f5f5',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#f5f5f5',
  },
  loadingText: {
    marginTop: 10,
    fontSize: 16,
    color: '#666',
  },
  header: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    backgroundColor: '#fff',
    padding: 15,
    paddingTop: 50,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  titleContainer: {
    flex: 1,
  },
  title: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#333',
  },
  wsStatusContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginTop: 5,
  },
  wsStatusDot: {
    width: 8,
    height: 8,
    borderRadius: 4,
    marginRight: 5,
  },
  wsStatusText: {
    fontSize: 12,
    color: '#666',
  },
  logoutButton: {
    padding: 5,
  },
  logoutText: {
    color: '#FF3B30',
    fontSize: 14,
    fontWeight: '600',
  },
  messagesCard: {
    backgroundColor: '#fff',
    marginHorizontal: 15,
    marginTop: 15,
    padding: 15,
    borderRadius: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
    borderLeftWidth: 4,
    borderLeftColor: '#4CAF50',
  },
  messageItem: {
    marginTop: 10,
    padding: 10,
    backgroundColor: '#f9f9f9',
    borderRadius: 8,
    borderLeftWidth: 2,
    borderLeftColor: '#4CAF50',
  },
  messageHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 5,
  },
  messageIndex: {
    fontSize: 12,
    fontWeight: 'bold',
    color: '#4CAF50',
  },
  messageTime: {
    fontSize: 11,
    color: '#999',
  },
  messageUserRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
    paddingBottom: 8,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
  },
  messageUserLabel: {
    fontSize: 12,
    fontWeight: '600',
    color: '#666',
    marginRight: 8,
  },
  messageUserValue: {
    fontSize: 12,
    color: '#007AFF',
    fontWeight: '500',
    flex: 1,
  },
  messageContent: {
    fontSize: 14,
    color: '#333',
    lineHeight: 20,
  },
  currentLocationCard: {
    backgroundColor: '#fff',
    margin: 15,
    padding: 15,
    borderRadius: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  cardTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 10,
  },
  infoRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 5,
  },
  infoLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '600',
  },
  infoValue: {
    fontSize: 14,
    color: '#333',
    flex: 1,
    textAlign: 'right',
  },
  checkInId: {
    fontSize: 11,
    color: '#999',
  },
  buttonContainer: {
    flexDirection: 'row',
    paddingHorizontal: 15,
    gap: 10,
    marginBottom: 15,
  },
  checkInButton: {
    flex: 1,
    backgroundColor: '#007AFF',
    paddingVertical: 15,
    borderRadius: 10,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  checkInButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  refreshButton: {
    backgroundColor: '#fff',
    paddingVertical: 15,
    paddingHorizontal: 20,
    borderRadius: 10,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  refreshButtonText: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
  },
  sayHiButton: {
    flex: 1,
    backgroundColor: '#4CAF50',
    paddingVertical: 15,
    borderRadius: 10,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 3,
  },
  sayHiButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
  buttonDisabled: {
    opacity: 0.6,
  },
  listContainer: {
    flex: 1,
    paddingHorizontal: 15,
  },
  section: {
    marginBottom: 20,
  },
  sectionTitle: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: 10,
  },
  checkInItem: {
    backgroundColor: '#fff',
    padding: 15,
    borderRadius: 10,
    marginBottom: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    elevation: 2,
  },
  realtimeItem: {
    borderLeftWidth: 4,
    borderLeftColor: '#4CAF50',
  },
  checkInHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 10,
    paddingBottom: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  checkInLabel: {
    fontSize: 14,
    fontWeight: 'bold',
    color: '#007AFF',
  },
  checkInDate: {
    fontSize: 12,
    color: '#999',
  },
  emptyText: {
    textAlign: 'center',
    color: '#999',
    marginTop: 20,
    fontSize: 14,
  },
});

