# 📍 TENARAI Blueprint - Sample Mobile App (Expo + React Native)

O2Oモバイル向けバックエンド（AWS）を利用したシンプルな位置情報TENARAI Blueprint - Sample Mobile Appです。

## 🎯 機能

### Phase 1（基本機能）
- ✅ ユーザー認証（Amazon Cognito）
- ✅ チェックイン記録（API Gateway + Lambda + DynamoDB）
- ✅ チェックイン履歴表示
- ✅ チェックイン情報のテキスト表示（緯度・経度・ユーザーID）

### Phase 2（拡張機能）✨
- ✅ **リアルタイム更新（WebSocket）**
- ✅ **他のユーザーのチェックインをリアルタイム表示**
- ✅ **WebSocket接続状態インジケーター**
- 🔄 近くのユーザーのチェックイン表示（位置情報フィルター）

## 📋 必要要件

### 開発環境
- Node.js 18.x 以上
- npm または yarn
- **Expo CLI** (自動インストール)
- スマートフォン（実機テスト用）
  - iOS: iPhone（iOS 13以上）
  - Android: Android端末（Android 6.0以上）

### ⭐ Expoの利点
- ✅ **XcodeやAndroid Studio不要**（開発時）
- ✅ **Expo Goアプリでスマホから即テスト可能**
- ✅ **QRコードをスキャンするだけ**
- ✅ **環境構築が超簡単**

### AWSリソース
- Amazon Cognito User Pool
- API Gateway（REST API）
- **API Gateway（WebSocket API）** ✨ Phase 2
- **AWS Lambda（WebSocket Handler + Lambda Authorizer）** ✨ Phase 2
- Amazon DynamoDB

## 🚀 セットアップ手順

### 1. 依存関係のインストール

```bash
npm install
```

**たったこれだけ！** XcodeやAndroid Studioのインストールは不要です 🎉

### 2. バックエンド設定 ⭐ 重要 ⭐

`config/backend.js` を編集して、AWSのエンドポイント情報を設定します。

```javascript
const BACKEND_CONFIG = {
  cognito: {
    // Cognito User Pool ID
    // 確認方法: AWSコンソール > Cognito > User Pool > 一般設定
    userPoolId: 'us-east-1_xxxxxxxxx',

    // Cognito App Client ID
    // 確認方法: AWSコンソール > Cognito > User Pool > アプリクライアント
    clientId: '1234567890abcdefghijklmnop',

    // AWSリージョン
    region: 'us-east-1',
  },

  api: {
    // API GatewayのベースURL
    // 確認方法: AWSコンソール > API Gateway > ステージ
    // 例: https://xxxxxxxxxx.execute-api.us-east-1.amazonaws.com/prod
    baseUrl: 'https://xxxxxxxxxx.execute-api.us-east-1.amazonaws.com/prod',

    endpoints: {
      createCheckIn: '/checkin',
      getCheckIns: '/checkins',
      getNearbyCheckIns: '/checkins/nearby',
    },
  },

  // ✨ Phase 2: WebSocket設定
  websocket: {
    // WebSocket API URL
    // 確認方法: AWSコンソール > API Gateway > WebSocket API > ステージ
    // 例: wss://xxxxxxxxxx.execute-api.us-east-1.amazonaws.com/prod
    url: 'wss://xxxxxxxxxx.execute-api.us-east-1.amazonaws.com/prod',

    reconnect: {
      enabled: true,
      maxAttempts: 5,
      interval: 3000,
    },

    heartbeat: {
      enabled: true,
      interval: 30000,
    },
  },

  settings: {
    // リアルタイム更新を有効化（Phase 2）
    enableRealtime: true,
    searchRadiusKm: 5,
    historyLimit: 50,
  },
};
```

### 3. アプリの起動

#### 方法1: Expo Goアプリで実機テスト（推奨！）

1. **スマートフォンにExpo Goをインストール**
   - iOS: [App Store](https://apps.apple.com/app/expo-go/id982107779)
   - Android: [Google Play](https://play.google.com/store/apps/details?id=host.exp.exponent)

2. **開発サーバーを起動**
   ```bash
   npm start
   ```

3. **QRコードをスキャン**
   - iOS: カメラアプリでQRコードをスキャン
   - Android: Expo Goアプリ内でスキャン

4. **アプリが自動的に起動！** 🎉

#### 方法2: エミュレーターで実行

```bash
# Androidエミュレーター
npm run android

# iOSシミュレーター（Macのみ）
npm run ios
```

### 4. 開発のヒント

#### ホットリロード
- ファイルを保存すると自動的にアプリが更新されます
- Expo Goアプリで「r」キーを押すと手動リロード

#### デバッグ
- 開発サーバーの画面で「j」キーを押すとデバッガーが開きます
- `console.log()` の出力がターミナルに表示されます

#### トラブルシューティング
```bash
# キャッシュをクリアして再起動
npx expo start -c

# node_modulesを再インストール
rm -rf node_modules package-lock.json
npm install
```

## 📁 プロジェクト構造

```
CheckInApp/
├── app/                         # 📱 画面・ルーティング（Expo Router）
│   ├── _layout.js              # ルートレイアウト（認証制御）
│   ├── index.js                # メイン画面（チェックイン）
│   └── login.js                # ログイン・登録画面
│
├── config/                      # ⚙️ 設定
│   └── backend.js              # バックエンド設定 ⭐重要
│
├── services/                    # 🔧 ビジネスロジック
│   ├── auth.js                 # Cognito認証サービス
│   ├── api.js                  # REST APIクライアント
│   └── websocket.js            # ✨ WebSocketサービス（Phase 2）
│
├── assets/                      # 🖼️ 画像・アイコン
├── lambda-layer/               # AWS Lambda Layer
├── app.json                    # Expo設定
├── package.json
└── README.md
```

### 主要ファイルの説明

| ファイル                | 説明                             |
| ----------------------- | -------------------------------- |
| `app/_layout.js`        | アプリ全体のレイアウトと認証制御 |
| `app/index.js`          | メイン画面（チェックイン機能）   |
| `app/login.js`          | ログイン・ユーザー登録画面       |
| `config/backend.js`     | AWS設定（編集必須） ⭐            |
| `services/auth.js`      | Cognito認証ロジック              |
| `services/api.js`       | REST API通信                     |
| `services/websocket.js` | WebSocket通信（Phase 2）         |

## 🔧 バックエンドAPIの要件

### POST `/checkin`
チェックインを作成

**リクエスト:**
```json
{
  "latitude": 35.6812,
  "longitude": 139.7671,
  "memo": "チェックイン",
  "timestamp": "2025-11-15T10:00:00Z"
}
```

**レスポンス:**
```json
{
  "checkInId": "uuid-1234",
  "userId": "user123",
  "latitude": 35.6812,
  "longitude": 139.7671,
  "timestamp": "2025-11-15T10:00:00Z"
}
```

### GET `/checkins?limit=50`
チェックイン履歴を取得

**レスポンス:**
```json
{
  "items": [
    {
      "checkInId": "uuid-1234",
      "userId": "user123",
      "latitude": 35.6812,
      "longitude": 139.7671,
      "memo": "チェックイン",
      "timestamp": "2025-11-15T10:00:00Z"
    }
  ]
}
```

### WebSocket `/` (Phase 2)
リアルタイム通知を受信

**接続時:**
```json
{
  "action": "sendmessage",
  "data": {
    "type": "locationUpdate",
    "latitude": 35.6812,
    "longitude": 139.7671,
    "memo": "チェックイン"
  }
}
```

**受信メッセージ:**
```json
{
  "event": "newCheckIn",
  "data": {
    "checkInId": "uuid-5678",
    "userId": "user456",
    "latitude": 35.6815,
    "longitude": 139.7675,
    "timestamp": "2025-11-15T09:30:00Z"
  }
}
```

## 📱 アプリの使い方

### 1. ユーザー登録
1. アプリを起動
2. 「アカウントをお持ちでない方は こちら」をタップ
3. メールアドレス、パスワード（8文字以上）を入力
4. 「登録」をタップ
5. メールに届いた確認コードを入力

### 2. ログイン
1. メールアドレスとパスワードを入力
2. 「ログイン」をタップ

### 3. チェックイン
1. メイン画面で「チェックイン」ボタンをタップ
2. 現在地（緯度・経度）が自動的にチェックインされます
3. チェックイン履歴に表示されます

### 4. リアルタイム通知（Phase 2）
1. 画面上部のインジケーターで接続状態を確認
   - 🟢 緑：リアルタイム接続中
   - 🟠 オレンジ：接続中...
2. 他のユーザーがチェックインすると自動的に通知が届きます
3. リアルタイムチェックインセクションに表示されます

## 🎨 カスタマイズ

### リアルタイム機能の無効化
`config/backend.js` で設定：

```javascript
settings: {
  enableRealtime: false,  // WebSocket機能を無効化
}
```

### 検索範囲の変更
```javascript
settings: {
  searchRadiusKm: 10,  // 10kmに変更
}
```

### 履歴取得件数の変更
```javascript
settings: {
  historyLimit: 100,  // 100件に変更
}
```

## 🐛 トラブルシューティング

### バックエンドに接続できない
1. ✅ `config/backend.js` の設定を確認
2. ✅ API Gatewayが正しくデプロイされているか確認
3. ✅ CORS設定が正しいか確認
4. ✅ ネットワーク接続を確認

### 位置情報が取得できない
1. ✅ アプリの位置情報権限を確認
2. ✅ デバイスの位置情報サービスがONか確認
3. ✅ デフォルト位置（東京）にフォールバックされます

### ログインできない
1. ✅ Cognito User Poolの設定を確認
2. ✅ パスワードは8文字以上か確認
3. ✅ メール確認が完了しているか確認

### WebSocketに接続できない（Phase 2）
1. ✅ WebSocket URLが `wss://` で始まっているか確認
2. ✅ AWS API Gateway WebSocket APIがデプロイされているか確認
3. ✅ Lambda関数が正しく設定されているか確認
4. ✅ `config/backend.js` の `enableRealtime` が `true` か確認

### Expo Goで起動しない
```bash
# キャッシュをクリアして再起動
npx expo start -c

# 依存関係を再インストール
rm -rf node_modules package-lock.json
npm install
```

## 🔐 セキュリティ

- パスワードは暗号化されてCognitoに保存されます
- JWTトークンは `AsyncStorage` に安全に保存されます
- APIリクエストには認証トークンが自動で付与されます
- WebSocket接続にもJWT認証が必要です

## 📝 技術スタック

| カテゴリ           | 技術                 | バージョン |
| ------------------ | -------------------- | ---------- |
| **フレームワーク** | Expo                 | SDK 54     |
| **UI**             | React                | 19.1.0     |
| **モバイル**       | React Native         | 0.81.5     |
| **ルーティング**   | Expo Router          | ~6.0       |
| **認証**           | Amazon Cognito       | -          |
| **API**            | API Gateway + Lambda | -          |
| **データベース**   | DynamoDB             | -          |
| **リアルタイム**   | WebSocket API        | Phase 2    |

## 📝 Phase 2 WebSocket機能の詳細

### 機能概要

1. **自動接続**
   - アプリ起動時に自動的にWebSocketサーバーに接続
   - JWT認証による安全な接続

2. **リアルタイム通知**
   - 他のユーザーのチェックインを即座に受信
   - 通知ポップアップで詳細を確認

3. **自動再接続**
   - 接続が切れた場合、自動的に再接続（最大5回）
   - 再接続間隔は段階的に増加

4. **ハートビート**
   - 30秒ごとに接続維持のためのpingを送信
   - 接続の安定性を確保

### WebSocket設定

`config/backend.js` で詳細設定が可能：

```javascript
websocket: {
  reconnect: {
    enabled: true,        // 自動再接続
    maxAttempts: 5,       // 最大再接続回数
    interval: 3000,       // 再接続間隔（ミリ秒）
  },
  heartbeat: {
    enabled: true,        // ハートビート
    interval: 30000,      // 間隔（30秒）
  },
}
```

## 📄 ライセンス

MIT License

## 🤝 サポート

問題が発生した場合は、以下を確認してください：
1. バックエンド設定（`config/backend.js`）
2. AWS側のリソース設定
3. ネットワーク接続
4. Expo Goアプリが最新版か確認

---

**作成日:** 2025年11月13日
**最終更新:** 2025年11月15日
**バージョン:** 3.0.0 (Expo Router対応 + Phase 2 WebSocket)
**技術スタック:** Expo SDK 54 + React 19.1 + React Native 0.81 + Expo Router 6 + WebSocket API
